"use client";
import Newsletter from "@/components/homepage/Newsletter";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { CircleArrowRight, Loader, Mail, Phone } from "lucide-react";
import Image from "next/image";
import React, { useState } from "react";
import { z } from "zod";

const formSchema = z.object({
  email: z.string().min(2).max(50),
  phone: z.string().min(2),
  message: z.string().optional(),
  name: z.string(),
});

import { zodResolver } from "@hookform/resolvers/zod";
import { useForm } from "react-hook-form";
import { Button } from "@/components/ui/button";
import { toast } from "sonner";

const OrderPage = () => {
  const [isLoading, setIsLoading] = useState(false);
  const form = useForm<z.infer<typeof formSchema>>({
    resolver: zodResolver(formSchema),
    defaultValues: {
      email: "",
      name: "",
      phone: "",
      message: "",
    },
  });

  const handleSubmit = async (values: z.infer<typeof formSchema>) => {
    try {
      setIsLoading(true);
      const { email, phone, message, name } = values;
      const res = await fetch("/api/contact", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          email,
          phone,
          name,
          message,
        }),
      });

      const data = await res.json();
      if (data.success)
        toast.success("ধন্যবাদ! আমরা খুব শীঘ্রই আপনার সাথে যোগাযোগ করব।");
      setIsLoading(false);
    } catch (err) {
      setIsLoading(false);
      toast.error(
        "দুঃখিত, বার্তা পাঠাতে ব্যর্থ হয়েছে। অনুগ্রহ করে হোয়াটসঅ্যাপ বা ইমেলের মাধ্যমে যোগাযোগ করুন।"
      );
    }
  };

  return (
    <div className="min-h-screen relative overflow-hidden">
      <div className="md:my-10 my-5 grid grid-cols-1 xl:grid-cols-2 gap-8 my-container items-center">
        <Form {...form}>
          <form onSubmit={form.handleSubmit(handleSubmit)}>
            <div className="text-black md:space-y-4 space-y-2">
              <h2 className="xl:text-5xl lg:text-4xl md:text-3xl text-2xl font-semibold xl:leading-16 md:pb-4 pb-2">
                আপনি কি আমাদের Onebite Premium ক্রয় করতে চান?
              </h2>
              <div className="flex md:flex-row flex-col md:gap-4 gap-2 w-full">
                <div className="bg-[#078B001A] rounded-lg p-5 gap-3 flex flex-col justify-center w-full py-2 min-h-[100px]">
                  <p className="flex text-gray font-medium text-base items-center gap-2">
                    <Mail /> ইমেইল করুন
                  </p>
                  <h3 className="text-black font-medium lg:text-2xl text-lg">
                    contact@oneheartbd.com
                  </h3>
                </div>
                <div className="bg-[#FF4A001A] rounded-lg flex flex-col justify-center gap-3 p-5 py-2 w-full min-h-[100px]">
                  <p className="flex text-gray font-medium text-base items-center gap-2">
                    <svg
                      width="21"
                      height="21"
                      viewBox="0 0 21 21"
                      fill="none"
                      xmlns="http://www.w3.org/2000/svg"
                      xmlnsXlink="http://www.w3.org/1999/xlink"
                    >
                      <rect
                        x="0.852539"
                        y="0.345001"
                        width="20"
                        height="20"
                        fill="url(#pattern0_238_191)"
                      />
                      <defs>
                        <pattern
                          id="pattern0_238_191"
                          patternContentUnits="objectBoundingBox"
                          width="1"
                          height="1"
                        >
                          <use
                            xlinkHref="#image0_238_191"
                            transform="scale(0.00195312)"
                          />
                        </pattern>
                        <image
                          id="image0_238_191"
                          width="512"
                          height="512"
                          preserveAspectRatio="none"
                          xlinkHref="data:image/png;base64,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"
                        />
                      </defs>
                    </svg>
                    কল করুন
                  </p>
                  <h3 className="text-black font-medium lg:text-2xl text-lg">
                    +8801754123456
                  </h3>
                </div>
              </div>
              <div className="flex md:flex-row flex-col gap-4 w-full">
                <FormField
                  control={form.control}
                  name="name"
                  render={({ field }) => (
                    <FormItem className="w-full">
                      <FormControl>
                        <Input
                          placeholder="নাম *"
                          {...field}
                          className="w-full"
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="email"
                  render={({ field }) => (
                    <FormItem className="w-full">
                      <FormControl>
                        <Input
                          placeholder="ইমেইল *"
                          {...field}
                          className="w-full"
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>
              <FormField
                control={form.control}
                name="phone"
                render={({ field }) => (
                  <FormItem>
                    <FormControl>
                      <Input placeholder="আপনার ফোন নম্বর *" {...field} />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="message"
                render={({ field }) => (
                  <FormItem>
                    <FormControl>
                      <Textarea
                        {...field}
                        className="min-h-36"
                        placeholder="একটা বার্তা লিখুন..."
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <Button
                disabled={isLoading}
                className="hind-siliguri-semibold rounded-xl justify-center text-sm text-white cursor-pointer h-[46px] flex items-center"
              >
                {!isLoading ? (
                  <span className="flex gap-2 items-center">
                    সাবমিট করুন <CircleArrowRight size={20} />
                  </span>
                ) : (
                  <span className="flex gap-2 items-center">
                    পাঠানো হচ্ছে <Loader className="animate-spin" size={20} />
                  </span>
                )}
              </Button>
            </div>
          </form>
        </Form>
        <div>
          <Image
            className="w-full h-full rounded-4xl object-cover"
            src={"/banner/section-1-extra.png"}
            height={550}
            width={521}
            alt="secton-1"
          />
        </div>
        {/* <Image
          className="z-10 absolute h-[550px] top-16 w-[550px] right-[100px] object-cover rounded-l-4xl"
          src={"/banner/section-1-extra.png"}
          height={550}
          width={521}
          alt="secton-1"
        /> */}
      </div>
      <Newsletter />
    </div>
  );
};

export default OrderPage;
